<?php
/* ──────────────────────────  boilerplate  ────────────────────────── */
declare(strict_types=1);
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Authorization, Content-Type');

require_once 'config.php';                     //  <-   db creds + UPLOAD_DIR

function jexit(int $code, string $msg): void {
    http_response_code($code);
    echo json_encode(['status' => 'error', 'message' => $msg], JSON_UNESCAPED_SLASHES);
    exit();
}
/* ─────────────────────────  happy-path  ─────────────────────────── */
try {

    /* 1 ─ verify method */
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        jexit(405, 'Only POST allowed');
    }

    /* 2 ─ check auth header */
    $headers = getallheaders();
    if (
        empty($headers['Authorization']) ||
        !preg_match('/Bearer\s+(\S+)/', $headers['Authorization'], $m)
    ) {
        jexit(401, 'Missing bearer token');
    }
    $token = $m[1];

    /* 3 ─ db connect + token lookup */
    $conn = getDbConnection() ?: jexit(500, 'DB connection failed');
    $stmt = $conn->prepare('SELECT user_id FROM users WHERE token = ?');
    if (!$stmt) {
        error_log('Prepare failed: '.$conn->error);
        jexit(500, 'Server error');
    }
    $stmt->bind_param('s', $token);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();
    if (!$row) jexit(401, 'Invalid token');
    $userId = (int)$row['user_id'];

    /* 4 ─ input validation */
    $houseId    = $_POST['house_id']   ?? null;
    $imageType  = $_POST['image_type'] ?? null;         // NEW
    $desc       = trim($_POST['description'] ?? '');

    if (!ctype_digit((string)$houseId)) {
        jexit(400, 'house_id must be numeric');
    }
    if (!in_array($imageType, ['passport','id_front','id_back'], true)) {
        jexit(400, 'image_type must be passport | id_front | id_back');
    }

    if (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
        jexit(400, 'No image uploaded');
    }
    $file = $_FILES['image'];

    /* 5 ─ mime & resize (keep <= 500 × 500) */
    $mime = mime_content_type($file['tmp_name']);
    if (!in_array($mime, ['image/jpeg','image/png'], true)) {
        jexit(400, 'Only JPG/PNG allowed');
    }
    /* very small GD-based resize */
    [$w,$h] = getimagesize($file['tmp_name']);
    $scale  = min(500/$w, 500/$h, 1);             // never upscale
    if ($scale < 1) {
        $newW = (int)($w*$scale);
        $newH = (int)($h*$scale);
        $src  = $mime==='image/png' ? imagecreatefrompng($file['tmp_name'])
                                    : imagecreatefromjpeg($file['tmp_name']);
        $dst  = imagecreatetruecolor($newW,$newH);
        imagecopyresampled($dst,$src,0,0,0,0,$newW,$newH,$w,$h);
        ($mime==='image/png')
            ? imagepng($dst, $file['tmp_name'])
            : imagejpeg($dst, $file['tmp_name'], 90);
        imagedestroy($src); imagedestroy($dst);
    }

    /* 6 ─ build upload path  */
    $baseDir = rtrim($config['UPLOAD_DIR'] ?? (__DIR__.'/../uploads/'), '/').'/';
    $houseDir = $baseDir.'house_'.$houseId.'/';
    if (!is_dir($houseDir) && !mkdir($houseDir, 0755, true)) {
        error_log("mkdir failed: $houseDir");
        jexit(500, 'Server error');
    }
    $fileName = uniqid()."_".preg_replace('/\s+/', '_', basename($file['name']));
    $dest     = $houseDir.$fileName;
    if (!move_uploaded_file($file['tmp_name'], $dest)) {
        jexit(500, 'Could not save file');
    }

    /* 7 ─ insert metadata */
    $stmt = $conn->prepare(
        'INSERT INTO visitorimages (user_id, house_id, file_name, description, image_type)
         VALUES (?,?,?,?,?)'
    );
    if (!$stmt) {
        error_log('insert prepare: '.$conn->error);
        jexit(500, 'Server error');
    }
    $stmt->bind_param('iisss', $userId, $houseId, $fileName, $desc, $imageType);
    $stmt->execute();

    /* 8 ─ done -> respond */
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
    $url = $protocol.$_SERVER['HTTP_HOST'].'/uploads/house_'.$houseId.'/'.$fileName;

    echo json_encode(['status'=>'success','imageUrl'=>$url,'type'=>$imageType]);

} catch (Throwable $e) {
    /* final safety-net */
    error_log('fatal upload error: '.$e->getMessage());
    jexit(500, 'Unexpected server error');
}
