<?php
// guardianapi/visitor_stats.php
declare(strict_types=1);

// --- CORS & content type ---
header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

// Preflight
if (strcasecmp($_SERVER['REQUEST_METHOD'] ?? 'GET', 'OPTIONS') === 0) {
  http_response_code(204);
  exit;
}

require_once __DIR__ . '/config.php';
$conn = getDbConnection();

/* ---------- Safe helpers (unique names to avoid collisions) ---------- */
if (!function_exists('gp_send')) {
  function gp_send(array $arr, int $code=200){ http_response_code($code); echo json_encode($arr); exit; }
}
if (!function_exists('gp_get_bearer')) {
  function gp_get_bearer(): ?string {
    $h = null;
    if (function_exists('getallheaders')) {
      $all = getallheaders();
      $h = $all['Authorization'] ?? $all['authorization'] ?? null;
    }
    if (!$h && isset($_SERVER['HTTP_AUTHORIZATION'])) $h = $_SERVER['HTTP_AUTHORIZATION'];
    if ($h && preg_match('/Bearer\s+([A-Za-z0-9._\-]+)/', $h, $m)) return $m[1];
    if (!empty($_GET['token'])) return (string)$_GET['token']; // legacy support
    return null;
  }
}
if (!function_exists('gp_verify_token')) {
  function gp_verify_token(mysqli $conn, ?string $token): ?array {
    if (!$token) return null;
    // admin
    if ($stmt = $conn->prepare("SELECT admin_id AS id, organisation_id FROM admin WHERE token=? AND is_active=1 LIMIT 1")) {
      $stmt->bind_param("s", $token);
      $stmt->execute(); $res = $stmt->get_result();
      if ($row = $res->fetch_assoc()) { $stmt->close(); return ['role'=>'admin','id'=>(int)$row['id'],'organisation_id'=>(int)$row['organisation_id'],'token'=>$token]; }
      $stmt->close();
    }
    // user
    if ($stmt = $conn->prepare("SELECT user_id AS id, organisation_id, house_id FROM users WHERE token=? LIMIT 1")) {
      $stmt->bind_param("s", $token);
      $stmt->execute(); $res = $stmt->get_result();
      if ($row = $res->fetch_assoc()) {
        $stmt->close();
        return [
          'role' => 'user',
          'id'   => (int)$row['id'],
          'organisation_id' => (int)$row['organisation_id'],
          'house_id' => isset($row['house_id']) ? (int)$row['house_id'] : null,
          'token' => $token
        ];
      }
      $stmt->close();
    }
    return null;
  }
}

/* ---------- Auth ----------
 * If routed through index.php: $GLOBALS['AUTH'] is already set.
 * If called directly: derive auth here.
 */
if (!isset($GLOBALS['AUTH'])) {
  $tok  = gp_get_bearer();
  $auth = gp_verify_token($conn, $tok);
  if ($auth) $GLOBALS['AUTH'] = $auth;
}
if (!isset($GLOBALS['AUTH'])) gp_send(['status'=>'error','message'=>'Unauthorized'], 401);

$auth      = $GLOBALS['AUTH'];
$role      = (string)($auth['role'] ?? '');
$actorOrg  = (int)($auth['organisation_id'] ?? 0);
$isSuper   = ($role === 'admin' && ($actorOrg === 0 || (defined('SUPERADMIN_ORG_ID') && $actorOrg === (int)SUPERADMIN_ORG_ID)));

/* ---------- Inputs ---------- */
$orgId    = isset($_GET['organisation_id']) && $_GET['organisation_id'] !== '' ? (int)$_GET['organisation_id'] : null;
$houseId  = isset($_GET['house_id'])        && $_GET['house_id']        !== '' ? (int)$_GET['house_id']        : null;
$from     = isset($_GET['from']) ? trim((string)$_GET['from']) : '';
$to       = isset($_GET['to'])   ? trim((string)$_GET['to'])   : '';

/* Clamp non-superadmin to own org */
if (!$isSuper) $orgId = $actorOrg;

/* Validate date range if provided */
$hasRange = false;
if ($from !== '' || $to !== '') {
  if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $from)) gp_send(['status'=>'error','message'=>'Invalid from date (YYYY-MM-DD)'],400);
  if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $to))   gp_send(['status'=>'error','message'=>'Invalid to date (YYYY-MM-DD)'],400);
  $hasRange = true;
}

/* Build WHERE (global if $orgId === null and superadmin) */
$where = [];
$types = '';
$params = [];
if ($orgId !== null) { $where[] = 'organisation_id = ?'; $types .= 'i'; $params[] = $orgId; }
if ($houseId !== null) { $where[] = 'house_id = ?'; $types .= 'i'; $params[] = $houseId; }
$whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

/* Query helpers */
$scalar = function(string $sql, string $types, array $params) use ($conn){
  $stmt = $conn->prepare($sql);
  if (!$stmt) return 0;
  if ($types !== '') $stmt->bind_param($types, ...$params);
  $stmt->execute();
  $res = $stmt->get_result();
  $n = 0;
  if ($res && ($row = $res->fetch_row())) $n = (int)$row[0];
  $stmt->close();
  return $n;
};
$groupStatus = function(string $extraWhere, string $types, array $params) use ($conn){
  $sql = "SELECT status, COUNT(*) AS n FROM visitor_records $extraWhere GROUP BY status";
  $stmt = $conn->prepare($sql);
  if (!$stmt) return [];
  if ($types !== '') $stmt->bind_param($types, ...$params);
  $stmt->execute();
  $res = $stmt->get_result();
  $out = [];
  if ($res) {
    while ($row = $res->fetch_assoc()) {
      $out[] = ['status' => (string)$row['status'], 'count' => (int)$row['n']];
    }
  }
  $stmt->close();
  return $out;
};

/* Scope for response */
$scope = [
  'organisation_id' => $orgId, // null => global
  'house_id'        => $houseId,
];
if ($hasRange) { $scope['from'] = $from; $scope['to'] = $to; }

/* ---------- Range mode (if from/to supplied) ---------- */
if ($hasRange) {
  $rangeWhere = $where;
  $rangeTypes = $types . 'ss';
  $rangeParams= array_merge($params, [$from, $to]);
  $extra = $rangeWhere ? ('WHERE '.implode(' AND ', $rangeWhere).' AND ') : 'WHERE ';

  $totalRange  = $scalar("SELECT COUNT(*) FROM visitor_records $extra DATE(created_at) BETWEEN ? AND ?", $rangeTypes, $rangeParams);
  $exitedRange = $scalar("SELECT COUNT(*) FROM visitor_records $extra DATE(created_at) BETWEEN ? AND ? AND COALESCE(is_exited,0)=1", $rangeTypes, $rangeParams);
  $notExitedR  = max(0, $totalRange - $exitedRange);
  $byStatusR   = $groupStatus(($whereSql ? "$whereSql AND " : "WHERE ") . "DATE(created_at) BETWEEN '$from' AND '$to'", $types, $params);

  gp_send([
    'status'        => 'success',
    'scope'         => $scope + ['mode'=>'range'],
    'range_totals'  => [
      'all'           => $totalRange,
      'exited'        => $exitedRange,
      'not_exited'    => $notExitedR,
    ],
    'by_status'     => $byStatusR,
  ]);
}

/* ---------- Live/global snapshot (not page sized) ---------- */

/* All-time (global or scoped) */
$totalAll   = $scalar("SELECT COUNT(*) FROM visitor_records $whereSql", $types, $params);
$exitedAll  = $scalar(
  "SELECT COUNT(*) FROM visitor_records " .
  ($whereSql ? "$whereSql AND " : "WHERE ") . " COALESCE(is_exited,0)=1",
  $types, $params
);
$notExitedAll = max(0, $totalAll - $exitedAll);

/* Active now (status) */
$activeNow = $scalar(
  "SELECT COUNT(*) FROM visitor_records $whereSql" . ($whereSql ? " AND " : " WHERE ") . " status='active'",
  $types, $params
);

/* Today (created_at) */
$typesToday  = $types . 'ss';
$paramsToday = array_merge($params, [date('Y-m-d'), date('Y-m-d')]);
$totalToday  = $scalar(
  "SELECT COUNT(*) FROM visitor_records " .
  ($where ? 'WHERE '.implode(' AND ', $where).' AND ' : 'WHERE ') .
  "DATE(created_at) BETWEEN ? AND ?",
  $typesToday, $paramsToday
);
$exitedToday = $scalar(
  "SELECT COUNT(*) FROM visitor_records " .
  ($where ? 'WHERE '.implode(' AND ', $where).' AND ' : 'WHERE ') .
  "DATE(created_at) BETWEEN ? AND ? AND COALESCE(is_exited,0)=1",
  $typesToday, $paramsToday
);

/* This month */
$totalMonth = $scalar(
  "SELECT COUNT(*) FROM visitor_records " .
  ($where ? 'WHERE '.implode(' AND ', $where).' AND ' : 'WHERE ') .
  "DATE_FORMAT(created_at,'%Y-%m') = DATE_FORMAT(CURDATE(),'%Y-%m')",
  $types, $params
);
$exitedMonth = $scalar(
  "SELECT COUNT(*) FROM visitor_records " .
  ($where ? 'WHERE '.implode(' AND ', $where).' AND ' : 'WHERE ') .
  "DATE_FORMAT(created_at,'%Y-%m') = DATE_FORMAT(CURDATE(),'%Y-%m') AND COALESCE(is_exited,0)=1",
  $types, $params
);

/* By status snapshot */
$byStatusNow = $groupStatus($whereSql, $types, $params);

/* Response */
gp_send([
  'status'       => 'success',
  'scope'        => $scope + ['mode' => ($orgId === null && $isSuper ? 'global' : 'scoped')],
  'totals'       => [
    'all_time'          => $totalAll,
    'exited_all_time'   => $exitedAll,
    'not_exited_all_time'=> $notExitedAll,
    'active_now'        => $activeNow,
    'today'             => $totalToday,
    'exited_today'      => $exitedToday,
    'month'             => $totalMonth,
    'exited_month'      => $exitedMonth,
  ],
  'by_status'    => $byStatusNow,
]);
