<?php
// guardianapi/visitor_search.php
declare(strict_types=1);
header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

require_once __DIR__ . '/config.php';
$conn = getDbConnection();

/* ---------------- Fallback auth (no router) ---------------- */

function vs_get_token(): ?string {
  // Prefer Authorization: Bearer <token>, fallback to ?token=
  $hdr = '';
  if (isset($_SERVER['HTTP_AUTHORIZATION']))       $hdr = $_SERVER['HTTP_AUTHORIZATION'];
  elseif (isset($_SERVER['Authorization']))        $hdr = $_SERVER['Authorization'];
  elseif (function_exists('apache_request_headers')) {
    $h = apache_request_headers();
    if (isset($h['Authorization'])) $hdr = $h['Authorization'];
  }
  if ($hdr && preg_match('/Bearer\s+([A-Za-z0-9]+)$/', $hdr, $m)) return $m[1];
  if (isset($_GET['token']) && $_GET['token'] !== '') return (string)$_GET['token'];
  return null;
}

function vs_try_admin_auth(mysqli $c, string $token): ?array {
  // Validate against admin table only (backoffice use case)
  $sql = "SELECT organisation_id, is_active FROM admin WHERE token=? LIMIT 1";
  $st  = $c->prepare($sql);
  if (!$st) return null;
  $st->bind_param("s", $token);
  $st->execute(); $res = $st->get_result();
  $row = $res ? $res->fetch_assoc() : null;
  $st->close();
  if (!$row) return null;
  if ((int)$row['is_active'] !== 1) return null;
  return ['role'=>'admin','organisation_id'=>(int)$row['organisation_id']];
}

// If router didn’t set AUTH, try to authenticate by token here.
if (!isset($GLOBALS['AUTH'])) {
  $tok = vs_get_token();
  if ($tok) {
    $a = vs_try_admin_auth($conn, $tok);
    if ($a) { $GLOBALS['AUTH'] = $a; }
  }
}

/* ---------------- Authorize ---------------- */

if (!isset($GLOBALS['AUTH'])) {
  http_response_code(401);
  echo json_encode(['status'=>'error','message'=>'Unauthorized']);
  exit;
}

$auth     = $GLOBALS['AUTH'];
$role     = (string)($auth['role'] ?? '');
$actorOrg = (int)($auth['organisation_id'] ?? 0);
$super    = ($role === 'admin') && (
              $actorOrg === 0 ||
              $actorOrg === 7 ||
              (defined('SUPERADMIN_ORG_ID') && $actorOrg === (int)SUPERADMIN_ORG_ID)
            );

/* ---------------- Inputs ---------------- */

$qParam = isset($_GET['search']) ? (string)$_GET['search']
        : (isset($_GET['q']) ? (string)$_GET['q']
        : (isset($_GET['id_plates']) ? (string)$_GET['id_plates'] : ''));

$qParam  = trim($qParam);
$orgParam= isset($_GET['organisation_id']) && $_GET['organisation_id'] !== '' ? (int)$_GET['organisation_id'] : null;
$limit   = isset($_GET['limit'])  && (int)$_GET['limit']  > 0 ? (int)$_GET['limit']  : 50;
$offset  = isset($_GET['offset']) && (int)$_GET['offset'] >= 0 ? (int)$_GET['offset'] : 0;

/* ---------------- WHERE ---------------- */

$where  = [];
$types  = '';
$params = [];

// Scope: superadmin can search all or a specific org; others clamped to their org
if ($super) {
  if ($orgParam !== null) { $where[]='vr.organisation_id=?'; $types.='i'; $params[]=$orgParam; }
} else {
  $where[]='vr.organisation_id=?'; $types.='i'; $params[]=$actorOrg;
}

// Search terms
if ($qParam !== '') {
  $like = '%'.$qParam.'%';
  $or   = ['vr.name LIKE ?','vr.phone_number LIKE ?','vr.id_plates LIKE ?'];
  $types  .= 'sss';
  $params[] = $like; $params[] = $like; $params[] = $like;
  if (ctype_digit($qParam)) { $or[]='vr.id=?'; $types.='i'; $params[]=(int)$qParam; }
  $where[] = '(' . implode(' OR ', $or) . ')';
}

/* ---------------- Query ---------------- */

$sql = "
  SELECT
    vr.id,
    vr.name,
    vr.phone_number,
    vr.id_plates,
    vr.status,
    vr.created_at,
    vr.house_id,
    vr.organisation_id,
    h.house_number
  FROM visitor_records vr
  LEFT JOIN houses h ON h.id = vr.house_id
  " . (count($where) ? "WHERE ".implode(' AND ',$where) : "") . "
  ORDER BY vr.created_at DESC
  LIMIT ? OFFSET ?
";
$types  .= 'ii';
$params[] = $limit; $params[] = $offset;

$st = $conn->prepare($sql);
if (!$st) {
  http_response_code(500);
  echo json_encode(['status'=>'error','message'=>'DB error: '.$conn->error]);
  exit;
}
$st->bind_param($types, ...$params);
$st->execute(); $res = $st->get_result();

$out = [];
while ($r = $res->fetch_assoc()) {
  $out[] = [
    'id'             => (int)$r['id'],
    'name'           => (string)$r['name'],
    'house_id'       => $r['house_id'] !== null ? (int)$r['house_id'] : null,
    'house_number'   => $r['house_number'] ?? null,
    'phone_number'   => $r['phone_number'],
    'id_plates'      => $r['id_plates'],
    'status'         => $r['status'],
    'created_at'     => $r['created_at'],
    'organisation_id'=> (int)$r['organisation_id'],
  ];
}
$st->close();

/* ---------------- Response ---------------- */

echo json_encode([
  'status' => 'success',
  'scope'  => [
    'actor_role'    => $role,
    'actor_org'     => $actorOrg,
    'effective_org' => $super ? ($orgParam ?? null) : $actorOrg,
  ],
  'q'               => $qParam,
  'count'           => count($out),
  'visitor_records' => $out
]);
