<?php
// uniform.php — Hardened upload endpoint with robust logging to error.log only

// ---------- Headers ----------
header("Content-Type: application/json; charset=utf-8");
header("Access-Control-Allow-Origin: *");
header("Vary: Origin");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Authorization, Content-Type, X-Requested-With");

// Preflight support
if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'OPTIONS') {
    http_response_code(204);
    exit;
}

// ---------- Error handling & logging (log-only) ----------
error_reporting(E_ALL);
ini_set('display_errors', '0'); // do not leak errors to clients
ini_set('log_errors', '1');
ini_set('error_log', __DIR__ . '/error.log'); // <= all PHP errors go here

// Small structured logger
function apilog(string $msg, array $ctx = []): void {
    $line = '[uniform] ' . $msg;
    if ($ctx) $line .= ' ' . json_encode($ctx, JSON_UNESCAPED_SLASHES);
    error_log($line);
}

// Convert warnings/notices to exceptions for consistent handling
set_error_handler(function ($severity, $message, $file, $line) {
    if (!(error_reporting() & $severity)) return false;
    throw new ErrorException($message, 0, $severity, $file, $line);
});

// Uncaught exceptions
set_exception_handler(function (Throwable $e) {
    apilog('UNCAUGHT', ['type'=>get_class($e), 'msg'=>$e->getMessage()]);
    if (!headers_sent()) {
        http_response_code(500);
        header("Content-Type: application/json; charset=utf-8");
    }
    echo json_encode(['status'=>'error','message'=>'Server error','code'=>'UNCAUGHT']);
});

// Fatal errors
register_shutdown_function(function () {
    $err = error_get_last();
    if ($err && in_array($err['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR], true)) {
        apilog('FATAL', ['type'=>$err['type'], 'msg'=>$err['message'], 'file'=>$err['file'], 'line'=>$err['line']]);
        if (!headers_sent()) {
            http_response_code(500);
            header("Content-Type: application/json; charset=utf-8");
        }
        echo json_encode(['status'=>'error','message'=>'Server error','code'=>'FATAL']);
    }
});

// Lightweight request id to correlate logs
$reqId = bin2hex(random_bytes(6));
header('X-Request-Id: ' . $reqId);
apilog('REQ', ['id'=>$reqId, 'method'=>($_SERVER['REQUEST_METHOD'] ?? ''), 'uri'=>($_SERVER['REQUEST_URI'] ?? '')]);

// ---------- Only POST ----------
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
    http_response_code(405);
    echo json_encode(['status'=>'error','message'=>'Invalid request method. Only POST is allowed.','code'=>'METHOD']);
    apilog('METHOD_NOT_ALLOWED', ['got'=>($_SERVER['REQUEST_METHOD'] ?? '')]);
    exit;
}

// ---------- DB ----------
require_once __DIR__ . '/config.php';
$conn = getDbConnection(); // expects mysqli

if (!$conn || !($conn instanceof mysqli) || $conn->connect_errno) {
    apilog('DB_CONNECT_FAIL', ['errno'=>$conn ? $conn->connect_errno : null, 'error'=>$conn ? $conn->connect_error : 'no conn']);
    http_response_code(500);
    echo json_encode(['status'=>'error','message'=>'Database connection failed','code'=>'DB_CONNECT']);
    exit;
}
$conn->set_charset('utf8mb4');

// ---------- Authorization (robust header fetch) ----------
function getAuthHeader(): ?string {
    // Works on Apache/FPM/CGI variants
    if (!empty($_SERVER['HTTP_AUTHORIZATION'])) return $_SERVER['HTTP_AUTHORIZATION'];
    if (!empty($_SERVER['Authorization'])) return $_SERVER['Authorization']; // some FastCGI
    if (function_exists('apache_request_headers')) {
        $hdrs = apache_request_headers();
        foreach ($hdrs as $k => $v) {
            if (strtolower($k) === 'authorization') return $v;
        }
    }
    return null;
}
$authHeader = getAuthHeader();
if (!$authHeader || !preg_match('/^Bearer\s+(\S+)/i', $authHeader, $m)) {
    http_response_code(401);
    echo json_encode(['status'=>'error','message'=>'Authorization token is missing or invalid.','code'=>'NO_TOKEN']);
    apilog('NO_TOKEN');
    exit;
}
$token = $m[1];

// Validate token
$stmt = $conn->prepare("SELECT user_id FROM guardusers WHERE token = ?");
if (!$stmt) {
    apilog('DB_PREP_FAIL', ['stage'=>'token_select','error'=>$conn->error]);
    http_response_code(500);
    echo json_encode(['status'=>'error','message'=>'Database error while preparing query.','code'=>'DB_PREP']);
    exit;
}
$stmt->bind_param('s', $token);
$stmt->execute();
$res = $stmt->get_result();
$user = $res ? $res->fetch_assoc() : null;
$stmt->close();

if (!$user) {
    http_response_code(401);
    echo json_encode(['status'=>'error','message'=>'Unauthorized: Invalid token.','code'=>'BAD_TOKEN']);
    apilog('BAD_TOKEN', ['tail'=>substr($token, -6)]);
    exit;
}
$userId = (int)$user['user_id'];

// ---------- Upload validations ----------
if (!isset($_FILES['image']) || !is_array($_FILES['image'])) {
    http_response_code(400);
    echo json_encode(['status'=>'error','message'=>'No file uploaded.','code'=>'NO_FILE']);
    apilog('NO_FILE');
    exit;
}

$file = $_FILES['image'];

// Handle PHP upload error codes explicitly
if ($file['error'] !== UPLOAD_ERR_OK) {
    $errMap = [
        UPLOAD_ERR_INI_SIZE   => 'File exceeds php.ini upload_max_filesize',
        UPLOAD_ERR_FORM_SIZE  => 'File exceeds MAX_FILE_SIZE',
        UPLOAD_ERR_PARTIAL    => 'Partial upload',
        UPLOAD_ERR_NO_FILE    => 'No file uploaded',
        UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
        UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
        UPLOAD_ERR_EXTENSION  => 'A PHP extension stopped the file upload',
    ];
    $msg = $errMap[$file['error']] ?? 'Unknown upload error';
    http_response_code(400);
    echo json_encode(['status'=>'error','message'=>"Upload error: $msg",'code'=>'UPLOAD_ERR']);
    apilog('UPLOAD_ERR', ['code'=>$file['error'], 'msg'=>$msg]);
    exit;
}

// Enforce size limit (10MB)
if (($file['size'] ?? 0) > 10 * 1024 * 1024) {
    http_response_code(400);
    echo json_encode(['status'=>'error','message'=>'File size exceeds 10MB limit.','code'=>'SIZE']);
    apilog('SIZE_LIMIT', ['size'=>$file['size']]);
    exit;
}

// Server-side MIME validation using finfo (do not trust $_FILES['type'])
$finfo = new finfo(FILEINFO_MIME_TYPE);
$mime  = $finfo->file($file['tmp_name']) ?: '';
$allowed = ['image/jpeg' => 'jpg', 'image/png' => 'png'];
if (!isset($allowed[$mime])) {
    http_response_code(400);
    echo json_encode(['status'=>'error','message'=>'Invalid file type. Only JPEG and PNG are allowed.','code'=>'MIME']);
    apilog('BAD_MIME', ['mime'=>$mime, 'client_type'=>($file['type'] ?? '')]);
    exit;
}

// ---------- Prepare upload directory ----------
$uploadDir = __DIR__ . "/uploads/user_$userId/"; // keep it inside guardianapi/uniform.php directory
if (!is_dir($uploadDir)) {
    if (!mkdir($uploadDir, 0755, true)) {
        http_response_code(500);
        echo json_encode(['status'=>'error','message'=>'Failed to create upload directory.','code'=>'MKDIR']);
        apilog('MKDIR_FAIL', ['dir'=>$uploadDir]);
        exit;
    }
}

// Sanitize original name and generate unique server name
$origName = (string)($file['name'] ?? 'file');
$origName = preg_replace('/[^a-zA-Z0-9._-]/', '_', $origName);
$serverName = sprintf('%s_%s.%s',
    date('Ymd_His'),
    bin2hex(random_bytes(6)),
    $allowed[$mime]
);
$destPath = $uploadDir . $serverName;

// Move file
if (!move_uploaded_file($file['tmp_name'], $destPath)) {
    http_response_code(500);
    echo json_encode(['status'=>'error','message'=>'Failed to move uploaded file.','code'=>'MOVE']);
    apilog('MOVE_FAIL', ['tmp'=>$file['tmp_name'], 'dest'=>$destPath]);
    exit;
}

// Optional description (trim + cap length)
$description = isset($_POST['description']) ? trim((string)$_POST['description']) : '';
if (strlen($description) > 1000) $description = substr($description, 0, 1000);

// ---------- Save metadata ----------
$stmt = $conn->prepare("INSERT INTO images (user_id, file_name, description) VALUES (?, ?, ?)");
if (!$stmt) {
    apilog('DB_PREP_FAIL', ['stage'=>'image_insert','error'=>$conn->error]);
    http_response_code(500);
    echo json_encode(['status'=>'error','message'=>'Database error while preparing insertion query.','code'=>'DB_PREP2']);
    exit;
}
$stmt->bind_param('iss', $userId, $serverName, $description);
if (!$stmt->execute()) {
    apilog('DB_EXEC_FAIL', ['stage'=>'image_insert','error'=>$stmt->error]);
    http_response_code(500);
    echo json_encode(['status'=>'error','message'=>'Failed to save image details in database.','code'=>'DB_EXEC']);
    $stmt->close();
    // best-effort cleanup of the file if DB save failed
    @unlink($destPath);
    exit;
}
$stmt->close();

// Build public-ish path relative to your web root (adjust if needed)
$publicPath = "/guardianapi/uploads/user_$userId/$serverName";

// ---------- Success ----------
http_response_code(200);
apilog('UPLOAD_OK', ['user_id'=>$userId, 'file'=>$serverName, 'size'=>$file['size'] ?? 0]);
echo json_encode([
    'status'    => 'success',
    'message'   => 'Image uploaded successfully.',
    'file_name' => $serverName,
    'file_path' => $publicPath,
    'requestId' => $reqId
]);
